// 初始化函数
function initialize() {
  chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === "batchHighlight" && Array.isArray(request.items)) {
      // 批量处理高亮
      batchHighlightTextInEditor(request.items);
    } else if (request.action === "cancelHighlight") {
      // 取消高亮
      batchHighlightTextInEditor([]);
    }
  });
  // 监听所有可能的编辑器事件
  const events = ["input", "focus"];
  // 处理 iframe 内的事件
  function setupIframeEvents(iframe) {
    try {
      const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
      events.forEach((eventType) => {
        iframeDoc.addEventListener(eventType, handleEditableArea, true);
      });
      // 特别处理 UEditor 的选区变化
      iframeDoc.addEventListener(
        "selectionchange",
        () => {
          handleEditableArea({ target: iframeDoc.body });
        },
        true
      );
    } catch (e) { }
  }
  // 监听主文档事件
  events.forEach((eventType) => {
    document.addEventListener(eventType, handleEditableArea, true);
  });

  // 监听选区变化
  document.addEventListener(
    "selectionchange",
    () => {
      const selection = window.getSelection();
      if (selection.rangeCount > 0) {
        const range = selection.getRangeAt(0);
        const container = range.commonAncestorContainer;
        const element =
          container.nodeType === 3 ? container.parentElement : container;
        handleEditableArea({ target: element });
      }
    },
    true
  );

  // 监听 iframe 的创建和变化
  const observer = new MutationObserver((mutations) => {
    mutations.forEach((mutation) => {
      mutation.addedNodes.forEach((node) => {
        if (node.tagName === "IFRAME") {
          node.addEventListener("load", () => setupIframeEvents(node));
          setupIframeEvents(node);
        }
      });
    });
  });

  observer.observe(document.body, {
    childList: true,
    subtree: true,
  });

  // 立即处理已存在的 iframe
  document.querySelectorAll("iframe").forEach(setupIframeEvents);
}

// 统一处理可编辑区域
function handleEditableArea(event) {
  const target = event.target;
  const editableElement = getActualEditableElement(target);

  if (editableElement) {
    // 检查是否已经有按钮
    const existingButtons = document.querySelectorAll(".precheck-btn");
    if (existingButtons.length > 0) {
      return; // 如果已经有按钮，直接返回
    }
    requestAnimationFrame(() => addCheckButton(editableElement));
  }
}

// 获取实际可编辑元素
function getActualEditableElement(element) {
  if (!element) return null;

  // 处理 UEditor 的特殊情况
  const ueditorFrame = element.closest(".edui-editor-body iframe");
  if (ueditorFrame) {
    try {
      return ueditorFrame.contentDocument.body;
    } catch (e) {
      return element;
    }
  }

  // 处理 UEditor 容器
  const ueditorContainer = element.closest(".edui-body-container");
  if (ueditorContainer) {
    return ueditorContainer;
  }

  // 处理普通 iframe
  if (element.tagName === "IFRAME") {
    try {
      const doc = element.contentDocument || element.contentWindow.document;
      return doc.body || element;
    } catch (e) {
      return element;
    }
  }

  // 向上查找最外层的可编辑容器
  let current = element;
  let topEditableElement = null;

  while (current && current !== document.body) {
    if (isEditableElement(current)) {
      topEditableElement = current;
    }
    current = current.parentElement;
  }

  return topEditableElement;
}
// 检查元素是否可编辑
function isEditableElement(element) {
  if (!element) return false;
  const tagName = element.tagName.toUpperCase();
  // 基础可编辑特性检查
  return (
    element.isContentEditable ||
    tagName === "TEXTAREA" ||
    element.tagName === "textarea" ||
    element.getAttribute("contenteditable") === "true"
  );
}

function addCheckButton(element) {
  // 获取真实的编辑区域元素
  const targetElement = getActualEditableElement(element);
  if (!targetElement) return;
  // 检查整个页面是否已经存在按钮
  const existingButton = targetElement.querySelector(".precheck-btn");
  if (existingButton) {
    return; // 如果已经有按钮，直接返回
  }
  targetElement
    .querySelectorAll(".precheck-btn-host")
    .forEach((el) => el.remove());
  targetElement
    .querySelectorAll(".precheck-btn")
    .forEach((btn) => btn.remove());
  // 创建按钮
  const button = document.createElement("img");
  button.className = "precheck-btn";
  button.src = chrome.runtime.getURL("images/checkbtn.png");
  //设置img属性 data-type
  button.setAttribute("data-type", "png");
  button.setAttribute("data-src", button.src);

  // 定义更新按钮位置函数
  const updateButtonPosition = () => {
    const rect = targetElement.getBoundingClientRect();

    // 计算按钮的基础位置
    let buttonLeft = rect.left + rect.width - 30;

    // 水平方向调整
    buttonLeft = Math.max(24, Math.min(buttonLeft, window.innerWidth - 24));

    // 垂直方向调整
    const buttonHeight = 24; // 按钮高度
    const visibleTop = 10; // 距离顶部的最小距离
    const visibleBottom = window.innerHeight - buttonHeight - 10; // 距离底部的最小距离

    let buttonTop;

    // 计算元素在可视区域内的部分
    const visibleTopEdge = Math.max(0, rect.top);
    const visibleBottomEdge = Math.min(window.innerHeight, rect.bottom);

    if (visibleBottomEdge <= visibleTopEdge) {
      // 元素完全不在可视区域内
      buttonTop = window.innerHeight / 2;
    } else {
      // 元素部分可见，将按钮放在可见部分的中间
      buttonTop = visibleTopEdge + (visibleBottomEdge - visibleTopEdge) / 2;
      // 确保按钮在可视区域内
      buttonTop = Math.max(visibleTop, Math.min(buttonTop, visibleBottom));
    }

    button.style.left = `${buttonLeft}px`;
    button.style.top = `${buttonTop}px`;
    button.style.transform = `translateY(-50%)`;
  };

  // 失焦处理
  const handleFocusOut = (e) => {
    // 检查新的焦点是否在编辑区域或按钮内
    const isInEditableArea =
      targetElement.contains(e.relatedTarget) ||
      e.relatedTarget === button ||
      (e.relatedTarget && e.relatedTarget.closest(".precheck-btn"));
    if (!isInEditableArea) {
      cleanup();
    }
  };
  targetElement.addEventListener("focusout", handleFocusOut);
  // 清理函数
  let cleanup = () => {
    if (updateButtonPosition) {
      window.removeEventListener("resize", updateButtonPosition);
      targetElement.removeEventListener("input", updateButtonPosition);
    }
    targetElement.removeEventListener("focusout", handleFocusOut);
    button.remove();
  };
  // 添加点击事件处理
  const handleClick = debounce(async (e) => {
    e.preventDefault();
    e.stopPropagation();
    let html = await cancelHighlightElement(targetElement);
    const text = await getElementContent(html);
      chrome.runtime.sendMessage({
        action: "openSidePanel",
        text: text.replace(/&nbsp;/g, " ").replace(/\r/g, " ").trim(),
        isTextarea: targetElement.tagName === "TEXTAREA" ? true : false,
      });
  }, 1000);
  // 添加mousedown事件监听点击事件
  button.addEventListener("mousedown", handleClick, true);
  // 判断targetElement是不是textarea或者是可编辑元素
  if (
    targetElement.tagName === "TEXTAREA" ||
    (targetElement.isContentEditable &&
      !targetElement.closest("iframe") &&
      !targetElement.ownerDocument.defaultView.frameElement) //排除编辑器容器
  ) {
    Object.assign(button.style, {
      width: "24px",
      height: "24px",
      cursor: "pointer",
      position: "fixed",
      zIndex: "2147483647",
      display: "block",
      objectFit: "contain",
      maxWidth: "24px",
      maxHeight: "24px",
      transition: "all 0.2s ease",
    });

    // 初始化按钮位置
    updateButtonPosition();

    // 添加事件监听
    window.addEventListener("resize", updateButtonPosition);
    targetElement.addEventListener("input", updateButtonPosition);
    document.body.appendChild(button);
  } else {
    // 创建一个 宿主 元素
    const host = document.createElement("p");
    host.className = "precheck-btn-host";

    // 创建 shadow root
    const shadow = host.attachShadow({ mode: "closed" });
    // 设置按钮样式
    Object.assign(button.style, {
      width: "24px",
      height: "24px",
      cursor: "pointer",
      display: "block",
      maxWidth: "24px",
      maxHeight: "24px",
      float: "right",
      zIndex: "2147483647",
    });
    // 将按钮添加到 shadow DOM 中
    shadow.appendChild(button);
    // 找到最后一个子元素
    targetElement.appendChild(host);
    // 修改清理函数
    let originalCleanup = cleanup;
    cleanup = () => {
      originalCleanup();
      if (host) {
        host?.remove();
      }
    };
  }
}
function getElementContent(element) {
  if (!element) return "";

  // 处理 iframe
  if (element.tagName === "IFRAME") {
    try {
      const doc = element.contentDocument || element.contentWindow.document;
      const editorBody = doc.querySelector(".edui-body-container") || doc.body;
      return editorBody.innerHTML;
    } catch (e) {
      return "";
    }
  }

  // 处理 UEditor 容器
  const ueditorContainer = element.closest(".edui-body-container");
  if (ueditorContainer) {
    return ueditorContainer.innerHTML;
  }

  // 处理普通输入框
  if (element.value !== undefined) return element.value;

  // 处理可编辑div等元素
  if (element.isContentEditable) {
    return element.innerHTML;
  }
  // 处理其他情况
  return element.innerHTML || element.textContent || "";
}
// 批量高亮处理函数
function batchHighlightTextInEditor(items) {
  // 1. 处理 iframe 类型
  const iframes = document.querySelectorAll("iframe");
  iframes.forEach(async(iframe) => {
    try {
      const actualElement = getActualEditableElement(iframe);
      let html = await cancelHighlightElement(actualElement);
      if (items?.length > 0) {
        const existingButtons =
          html.querySelectorAll(".precheck-btn-host");
        if (html && existingButtons?.length > 0) {
          batchHighlightElement(html, items);
        }
      } else {
        cancelHighlightElement(actualElement);
      }
    } catch (e) {
      console.warn("无法访问 iframe 内容", e);
    }
  });
  // 2. 处理 contenteditable 类型
  const editableElements = document.querySelectorAll("[contenteditable=true]");
  editableElements.forEach((element) => {
    const actualElement = getActualEditableElement(element);
    if (items?.length > 0) {
      if (actualElement) {
        batchHighlightElement(actualElement, items);
      }
    } else {
      cancelHighlightElement(actualElement);
    }
  });
}
// 批量高亮元素内容 iframe类富文本编辑器
function batchHighlightElement(element, items) {
  // 对于富文本编辑器，获取原始内容
  let originalContent = getElementContent(element);
  // 如果没有内容，直接返回
  if (!originalContent) return;
  // 创建一个Set来存储需要排除的索引
  let excludedIndices = new Set();
  // 获取所有的style标签和它们的文本范围
  const styleTags = regexData(originalContent, /style="([^"]*)".*?>/g);
  // 遍历数据，找出需要排除的索引
  items.forEach((item, index) => {
    if (item?.originWordStart?.length) {
      for (let [start, end] of item.originWordStart) {
        // 检查 start 和 end 是否有效
        if (start < 0 || end > originalContent.length || start > end) {
          continue;
        }
        // 检查 [start, end] 是否在styleTags的任意一个tag的范围内
        if (styleTags.some((tag) => start >= tag.start && end <= tag.end)) {
          excludedIndices.add(index);
          break;
        }
      }
    }
  });
  // 创建一个替换计划数组，记录每个替换项的起始位置、结束位置和替换内容
  const replacementPlan = [];
  // 首先，获取所有的<a>标签和它们的文本范围
  const aTags = regexData(originalContent, /<a\s+[^>]*>/g);
  // 遍历替换计划,不在标签内得添加markId
  items.forEach((item, index) => {
    if (item?.originWordStart?.length && item.word) {
      item.originWordStart?.forEach((el) => {
        let start = el[0];
        let end = el[1];
        if (start < 0 || end > originalContent.length || start > end) {
          return;
        }
        let contentSlice = originalContent.substring(start, end);
        // 检查这个片段是否在<a>标签内
        let inATag = false;
        for (let tag of aTags) {
          if (start >= tag.start && end <= tag.end) {
            inATag = true;
            break;
          }
        }
        // 不在style和<a>标签内
        if (!inATag) {
          replacementPlan.push({
            start: start,
            end: end,
            content: `<span name="showyujian" class="showyujian" textstyle  style="background-color:rgb(255, 215, 0)">${contentSlice}</span>`,
          });
        }
      });
    }
  });
  replacementPlan.sort((a, b) => a.start - b.start);
  // 执行替换计划
  for (let i = 0; i < replacementPlan.length; i++) {
    const step = replacementPlan[i];
    const newContentLength = step.content.length - (step.end - step.start);
    // 更新后续步骤的起始和结束位置
    for (let j = i + 1; j < replacementPlan.length; j++) {
      const nextStep = replacementPlan[j];
      if (nextStep.start > step.start) {
        nextStep.start += newContentLength;
      }
      if (nextStep.end > step.start) {
        nextStep.end += newContentLength;
      }
    }
    originalContent =
      originalContent.substring(0, step.start) +
      step.content +
      originalContent.substring(step.end);
  }
  element.innerHTML = originalContent;
}
//获取所有的tags标签和它们的文本范围
const regexData = (content, regex) => {
  const matches = [...content.matchAll(regex)]; // 将迭代器转换为数组
  const aTags = matches.map((match) => {
    const [fullMatch] = match; // match 数组的第一个元素是完整的匹配项
    const start = match.index; // 匹配项的起始索引
    const end = start + fullMatch.length; // 匹配项的结束索引（实际上是最后一个字符的索引加1）
    return {
      start,
      end,
    };
  });
  return aTags;
};
// 取消定位
const cancelHighlightElement = (actualElement) => {
  let elements = actualElement.querySelectorAll('span[name="showyujian"]');
  if (elements?.length === 0) {
    let list1 = actualElement.querySelectorAll(
      'span[style*="background-color: rgb(255, 215, 0)"]'
    );
    let lsit2 = actualElement.querySelectorAll(
      'span[style*="background-color: #FFD700"]'
    );
    elements = list1?.length > 0 ? list1 : lsit2;
  }
  elements.forEach((el) => {
    //直接取span中得文本
    el.outerHTML = el.textContent;
  });
  return actualElement;
};
function debounce(func, delay = 300) {
  let timer = null;
  return function (...args) {
    clearTimeout(timer);
    timer = setTimeout(() => {
      func.apply(this, args);
    }, delay);
  };
}
// 初始化
if (document.readyState === "loading") {
  document.addEventListener("DOMContentLoaded", initialize);
} else {
  initialize();
}
